/*
 * CurveMovementInfo.h
 *
 * Created 9/1/2009 By Johnny Huynh
 *
 * Version 00.00.01 9/1/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Contains information for a relative curve movement.
  */
 
 #ifndef CURVE_MOVEMENT_INFO_H
 #define CURVE_MOVEMENT_INFO_H
 
 template <typename T> class CurveMovementInfo;
 
 #include "global.h"
 
 #include "MovementInfo.h"
 
 /**
  * Class specification for CurveMovementInfo
  */
 template <typename T>
 class CurveMovementInfo : public MovementInfo<T>
 {
 // Data Members
 private:
    // 0 <= t <= 1
    // P(t) = at^2 + vt + p
    VECTOR3_TYPE _acceleration; // geographical coordinates vector pointing in a direction (magnitude matters)
    VECTOR3_TYPE _velocity; // geographical coordinates vector pointing in a direction (magnitude matters)
    
 // Local Functions
 public:
    CurveMovementInfo( const VECTOR3_TYPE& acceleration = VECTOR3_TYPE( ZERO, ZERO, ZERO ), 
                       const VECTOR3_TYPE& velocity = VECTOR3_TYPE( ZERO, ZERO, ZERO ),
                       const T distance = ONE, const double duration = 0.0 );
    CurveMovementInfo( const CurveMovementInfo<T>& curve_movement_info );
    virtual ~CurveMovementInfo();
    inline CurveMovementInfo<T>& operator=( const CurveMovementInfo<T>& curve_movement_info );
    inline const VECTOR3_TYPE& get_acceleration() const;
    inline void set_acceleration( const VECTOR3_TYPE& acceleration );
    inline const VECTOR3_TYPE& get_velocity() const;
    inline void set_velocity( const VECTOR3_TYPE& velocity );
    virtual inline double process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  * P(t) = (distance*(acceleration*t^2 + velocity*t)) + p
  */
 template <typename T>
 CurveMovementInfo<T>::CurveMovementInfo( const VECTOR3_TYPE& acceleration, const VECTOR3_TYPE& velocity,
                                          const T distance, const double duration )
                      : MovementInfo<T>( duration ),
                        _acceleration( distance*acceleration ),
                        _velocity( distance*velocity )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CurveMovementInfo<T>::CurveMovementInfo( const CurveMovementInfo<T>& curve_movement_info )
                      : MovementInfo<T>( curve_movement_info ),
                        _acceleration( curve_movement_info._acceleration ),
                        _velocity( curve_movement_info._velocity )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CurveMovementInfo<T>::~CurveMovementInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified CurveMovementInfo to this CurveMovementInfo.
  *
  * @param (const CurveMovementInfo<T>&) curve_movement_info
  * @return CurveMovementInfo<T>&
  */
 template <typename T>
 inline CurveMovementInfo<T>& CurveMovementInfo<T>::operator=( const CurveMovementInfo<T>& curve_movement_info )
 {
    MovementInfo<T>::operator=( curve_movement_info );
    _acceleration = curve_movement_info._acceleration;
    _velocity = curve_movement_info._velocity;
    
    return *this;
 }
 
 /**
  * get_acceleration() returns the curve movement acceleration.
  *
  * @return const VECTOR3_TYPE&
  */
 template <typename T>
 inline const VECTOR3_TYPE& CurveMovementInfo<T>::get_acceleration() const
 {
    return _acceleration;
 }
 
 /**
  * set_acceleration() sets the curve movement acceleration to the specified acceleration.
  *
  * @param (const VECTOR3_TYPE&) acceleration
  */
 template <typename T>
 inline void CurveMovementInfo<T>::set_acceleration( const VECTOR3_TYPE& acceleration )
 {
    _acceleration = acceleration;
 }
 
 /**
  * get_velocity() returns the curve movement velocity.
  *
  * @return const VECTOR3_TYPE&
  */
 template <typename T>
 inline const VECTOR3_TYPE& CurveMovementInfo<T>::get_velocity() const
 {
    return _velocity;
 }
 
 /**
  * set_velocity() sets the curve movement velocity to the specified velocity.
  *
  * @param (const VECTOR3_TYPE&) velocity
  */
 template <typename T>
 inline void CurveMovementInfo<T>::set_velocity( const VECTOR3_TYPE& velocity )
 {
    _velocity = velocity;
 }
 
 /**
  * process_movement() processes the movement specified by this MovementInfo, given the Object to process on,
  * the MoveInfo, and current time. The duration unprocessed returned is greater than zero if this MovementInfo
  * has finished processing, and there is time leftover.
  *
  * @param (Object<T>*) obj_Ptr
  * @param (MoveInfo<T>*) move_info_Ptr
  * @param (double) current_time
  * @return double - the time left unprocessed (i.e. a double greater than zero if this MovementInfo is done processing)
  */
 template <typename T>
 inline double CurveMovementInfo<T>::process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time )
 {  
    nassertr( obj_Ptr != NULL, MovementInfo<T>::get_duration() );
    
    double move_action_invoke_time( move_info_Ptr->get_time_move_action_was_invoked() );
    double last_process_time( move_info_Ptr->get_time_last_processed_move_task() );
    nassertr( last_process_time >= move_action_invoke_time, MovementInfo<T>::get_duration() );
    
    double duration( MovementInfo<T>::get_duration() );
    double elapse_time_since_invocation( current_time - move_action_invoke_time );
    
    if ( elapse_time_since_invocation > 0.0 && (_acceleration != VECTOR3_TYPE(ZERO,ZERO,ZERO) || _velocity != VECTOR3_TYPE(ZERO,ZERO,ZERO)) )
    {
        // gradually move for only the movement duration (measured in seconds)
        //if ( elapse_time_since_invocation > duration ) // if true, this will be the last time we process this movement_info
        
        VECTOR3_TYPE prev_location;
        if ( move_action_invoke_time > last_process_time ) // actually, this should never happen (see process_move() of "MoveInfo.h")
            prev_location = VECTOR3_TYPE( ZERO, ZERO, ZERO ); // prev_location = current location of character
        else
        {
            double prev_elapse_time_since_invocation( last_process_time - move_action_invoke_time );
            double prev_invocation_ratio( prev_elapse_time_since_invocation / duration );
            double prev_invocation_ratio_squared( prev_invocation_ratio * prev_invocation_ratio );
            prev_location = (_acceleration*prev_invocation_ratio_squared) + (_velocity*prev_invocation_ratio);
        }
        
        double invocation_ratio( (elapse_time_since_invocation >= duration ? 1.0 : elapse_time_since_invocation / duration) );
        double invocation_ratio_squared( invocation_ratio * invocation_ratio );
        
        // translate Object by specified distance units in the specified direction
        // (a*invocation_ratio^2 + v*invocation_ratio) - (a*prev_invocation_ratio^2 + v*prev_invocation_ratio)
        VECTOR3_TYPE current_location( (_acceleration*invocation_ratio_squared) + (_velocity*invocation_ratio) );
        obj_Ptr->translate( current_location - prev_location );
        // sometimes cause great floating-point error
    }
    
    if ( elapse_time_since_invocation > duration )
        return elapse_time_since_invocation - duration;
    else // elapse_time_since_invocation <= duration
        return 0.0;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 #endif // CURVE_MOVEMENT_INFO_H